--[[
	Bongos Quests
		quest tracker bar
--]]

local Bongos = LibStub('AceAddon-3.0'):GetAddon('Bongos3')
local Tracker = Bongos:NewModule('QuestTracker')
local Quests = Bongos:GetModule('Quests')

local TooltipMeta = getmetatable(CreateFrame('GameTooltip', nil, nil, 'GameTooltipTemplate')).__index
local L = LibStub('AceLocale-3.0'):GetLocale('Bongos3-Quests')
local CompletedQuests = {}


--[[ Startup ]]--

function Tracker:Load()
	local defaults = {
		point = "TOPRIGHT",
		y = -267.7,
		x = -14,
	}

	self.bar, isNew = Quests:CreateBar('quest tracker', defaults, nil, 'MEDIUM')
	if isNew then
		self.LoadBar(self.bar)
	end
end

function Tracker:Unload()
	self.bar:Destroy()
end

function Tracker:LoadBar()
	for k,v in pairs(Tracker) do self[k] = v end
	
	QuestWatchFrame:ClearAllPoints()
	QuestWatchFrame:SetPoint('TOPRIGHT', self, 'TOPRIGHT', 5, 7)

	self:Attach(QuestWatchFrame)
	self:RegisterEvent('QUEST_LOG_UPDATE', 'RegisterCompletedQuests')
	self:Hook(TooltipMeta, 'SetHyperlink', 'HookQuestLinks')
	
	self:Hook('QuestLog_Update', 'UpdatePanel')
	self:Hook('QuestWatch_Update', 'UpdateWatch')
	self:Hook('RemoveQuestWatch', 'UpdateSize')
	self:Hook('AddQuestWatch', 'AutoCompleteQuest')
end


--[[ Update Size ]]--

function Tracker:UpdateSize()
	if QuestWatchFrame:IsShown() then
		self:SetHeight(QuestWatchFrame:GetHeight() + (GetNumQuestWatches() - 1) * 4)
		self:SetWidth(QuestWatchFrame:GetWidth())
	else
		if Bongos:IsLocked() then
			self:SetHeight(0.1)
			self:SetWidth(0.1)
		else
			self:SetHeight(80)
			self:SetWidth(200)
		end
	end
end


--[[ Quest Management ]]--

function Tracker:IsLeveledQuest(index, isLink)
	return self.sets.showLevels and strmatch(isLink and index or GetQuestLink(index), 'quest:(%d+):(%d+)')
end

function Tracker:RegisterCompletedQuests()
	for i = 1, GetNumQuestLogEntries() do
		local title, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete, isDaily = GetQuestLogTitle(i)
		CompletedQuests[title] = isComplete == 1
	end
	self:UnregisterEvent('QUEST_LOG_UPDATE')
end

function Tracker:AutoCompleteQuest(i)
	if self.sets.autoHide and select(7, GetQuestLogTitle(i)) == 1 then
		local title = GetQuestLogTitle(i)
			
		if CompletedQuests[title] then
			UIErrorsFrame:AddMessage(format(L.AlreadyCompleted, title), 1, 0.10, 0.10)
			CompletedQuests[title] = true
		else
			UIErrorsFrame:AddMessage(format(L.Completed, title), 0.25, 0.75, 0.25)
		end
		
		return RemoveQuestWatch(i) or QuestWatch_Update() or true
	end
	self:UpdateSize()
end


--[[ Watch ]]--

function Tracker:UpdateWatch()
	local width = QuestWatchFrame:GetWidth() - 10
	local count = 1

	for i = 1, GetNumQuestWatches() do
		local index = GetQuestIndexForWatch(i)
		
		if index then
			if self:AutoCompleteQuest(index) then
				return
			end
			
			local numObjectives = GetNumQuestLeaderBoards(index)
			
			if self:IsLeveledQuest(index) then
				local title, level = GetQuestLogTitle(index)
					
				local string = _G["QuestWatchLine"..count]
				string:SetText(format("[%d] %s", level, title))
				width = max(string:GetWidth(), width)
			end
		
			count = count + numObjectives + 1
		end
	end
	
	QuestWatchFrame:SetWidth(width + 10)
	self:UpdateSize()
end


--[[ Panel ]]-- 

function Tracker:UpdatePanel()
	if not self.sets.showLevels then
		return
	end
	
	local numEntries = GetNumQuestLogEntries()
	for i = 1, QUESTS_DISPLAYED do	
		local index = FauxScrollFrame_GetOffset(QuestLogListScrollFrame) + i
		
		if index <= numEntries then
			local title, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete, isDaily = GetQuestLogTitle(index)
		
			if title and not isHeader and self:IsLeveledQuest(index) then
			
				local string = _G["QuestLogTitle"..i]
				string:SetText(format("  [%d] %s", level, title))
			
				if isComplete or isDaily then	
					local text = _G["QuestLogTitle"..i.."NormalText"]
					local check = _G["QuestLogTitle"..i.."Check"]
					local tag = _G["QuestLogTitle"..i.."Tag"]
				
					check:SetPoint("LEFT", string, "LEFT", max(250 - tag:GetWidth(), text:GetWidth()) + 24, 0)
				end
			
				_G["QuestLogTitle"..i.."GroupMates"]:SetPoint("LEFT", string, "LEFT", 13, 0)
			end
		end
	end
end


--[[ Links ]]--

function Tracker:HookQuestLinks(tooltip, link)
	if self:IsLeveledQuest(link, true) then
		local text = _G[tooltip:GetName()..'TextLeft1']
		local level = select(3, strsplit(':', link))
		
		text:SetText(format("%s [%d]", text:GetText() or "", level))
		tooltip:Show()
	end
end


--[[ Menu ]]--

function Tracker:CreateMenu()
	local menu = Bongos.Menu:Create(self.id)
	local panel = menu:AddLayoutPanel()
	
	self:CreateCheckButton(panel, 'ShowLevels')
	self:CreateCheckButton(panel, 'AutoHide')
	
	return menu
end

function Tracker:CreateCheckButton(panel, arg)
	local f = panel:CreateCheckButton(L[arg])
	arg = arg:gsub('%w+', function(s) return s:gsub('^%u', strlower) end)
	
	f:SetScript('OnShow', function() this:SetChecked(self.sets[arg]) end)
	f:SetScript('OnClick', function()
		self.sets[arg] = this:GetChecked()
		QuestWatch_Update()
		QuestLog_Update()
	end)
end